/**
 * Ask People Extension - Popup Script
 * Handles the extension popup UI and user interactions
 */

// Use browser API with chrome fallback for cross-browser compatibility
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

const contentEl = document.getElementById('content');

// State
let currentSession = null;
let feedbackState = null; // 'success' | 'error' | null

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
  await loadSession();
  render();
});

// Load current session from background
async function loadSession() {
  const response = await browserAPI.runtime.sendMessage({ type: 'GET_SESSION' });
  currentSession = response.session;
}

// Render the popup content
function render() {
  if (!currentSession || !currentSession.query) {
    renderNoSession();
  } else {
    renderActiveSession();
  }
}

// Render when no active query session
function renderNoSession() {
  contentEl.innerHTML = `
    <div class="no-session">
      <div class="no-session-icon">💤</div>
      <div class="no-session-text">
        No active search session.<br><br>
        Start a query on Ask People that doesn't have YAGO results to begin saving answers.
      </div>
    </div>
  `;
}

// Render active session UI
function renderActiveSession() {
  let feedbackHtml = '';
  if (feedbackState === 'success') {
    feedbackHtml = `
      <div class="feedback feedback-success">
        ✓ Answer saved!
      </div>
    `;
  } else if (feedbackState === 'error') {
    feedbackHtml = `
      <div class="feedback feedback-error">
        ✗ Failed to save. Please try again.
      </div>
    `;
  }

  contentEl.innerHTML = `
    <div class="session-active">
      <div class="query-text">${escapeHtml(currentSession.query)}</div>

      ${feedbackHtml}

      <button class="btn btn-primary" id="btnSavePage">
        💾 Save this page as answer
      </button>

      <div class="hint">
        Tip: Highlight text on any page and right-click to save it as an answer.
      </div>

      <button class="btn btn-done" id="btnDoneSearching">
        ✓ Done searching
      </button>
    </div>
  `;

  // Attach event listeners
  document.getElementById('btnSavePage')?.addEventListener('click', saveCurrentPage);
  document.getElementById('btnDoneSearching')?.addEventListener('click', endSession);
}

// Save current page as answer
async function saveCurrentPage() {
  try {
    // Get current tab info
    const [tab] = await browserAPI.tabs.query({ active: true, currentWindow: true });

    const response = await browserAPI.runtime.sendMessage({
      type: 'SAVE_PAGE_AS_ANSWER',
      url: tab.url,
      title: tab.title
    });

    if (response.success) {
      feedbackState = 'success';
    } else {
      feedbackState = 'error';
      console.error('Save failed:', response.error);
    }
    render();
  } catch (error) {
    feedbackState = 'error';
    console.error('Save error:', error);
    render();
  }
}

// End the current session
async function endSession() {
  await browserAPI.runtime.sendMessage({ type: 'END_SESSION' });
  currentSession = null;
  feedbackState = null;
  render();
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

