/**
 * Ask People Extension - Background Script
 * Handles message passing and state management for query sessions
 * Compatible with both Chrome and Firefox (Manifest V2)
 */

// Use browser API with chrome fallback for cross-browser compatibility
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

// Default API URL (can be configured)
const DEFAULT_API_URL = 'http://localhost:8000';

// Current query session state (in-memory, synced with storage)
let currentSession = {
  query: null,
  startedAt: null,
  apiUrl: DEFAULT_API_URL,
  source: null,
  intent: null
};

// Load session from storage on startup
async function loadSessionFromStorage() {
  try {
    const result = await browserAPI.storage.local.get('currentSession');
    if (result.currentSession && result.currentSession.query) {
      currentSession = result.currentSession;
      console.log('[Background] Loaded session from storage:', currentSession);
      // Update context menu
      browserAPI.contextMenus.update('saveSelectedAnswer', {
        title: `Save as answer to: "${currentSession.query.substring(0, 30)}${currentSession.query.length > 30 ? '...' : ''}"`
      });
    }
  } catch (err) {
    console.error('[Background] Failed to load session from storage:', err);
  }
}

// Save session to storage
async function saveSessionToStorage() {
  try {
    await browserAPI.storage.local.set({ currentSession: currentSession });
    console.log('[Background] Saved session to storage');
  } catch (err) {
    console.error('[Background] Failed to save session to storage:', err);
  }
}

// Initialize context menu for text selection
browserAPI.runtime.onInstalled.addListener(() => {
  browserAPI.contextMenus.create({
    id: 'saveSelectedAnswer',
    title: 'Save as answer to: "%s"',
    contexts: ['selection']
  });
});

// Load session on startup
loadSessionFromStorage();

// Broadcast message to all tabs
async function broadcastToAllTabs(message) {
  try {
    const tabs = await browserAPI.tabs.query({});
    for (const tab of tabs) {
      try {
        browserAPI.tabs.sendMessage(tab.id, message);
      } catch (err) {
        // Tab may not have content script, ignore
      }
    }
  } catch (err) {
    console.error('[Background] Failed to broadcast:', err);
  }
}

// Handle context menu clicks
browserAPI.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === 'saveSelectedAnswer' && currentSession.query) {
    const selectedText = info.selectionText;
    saveAnswer({
      type: 'text_answer',
      content: selectedText,
      title: `Selected from: ${tab.title}`,
      url: tab.url
    });
  }
});

// Listen for messages from content scripts and popup
browserAPI.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Background] Received message:', message.type);

  switch (message.type) {
    case 'START_QUERY_SESSION':
      // Website detected a query - start session for saving answers
      currentSession = {
        query: message.query,
        startedAt: Date.now(),
        apiUrl: message.apiUrl || DEFAULT_API_URL,
        source: message.source || null,
        intent: message.intent || null
      };
      console.log('[Background] Session started:', currentSession);
      // Update context menu title
      browserAPI.contextMenus.update('saveSelectedAnswer', {
        title: `Save as answer to: "${message.query.substring(0, 30)}${message.query.length > 30 ? '...' : ''}"`
      });
      // Persist to storage so session survives tab navigation
      saveSessionToStorage();
      // Broadcast to all tabs so they can show the session bar
      broadcastToAllTabs({ type: 'SESSION_STARTED', query: message.query });
      sendResponse({ success: true });
      break;

    case 'GET_SESSION':
      // Popup or content script wants current session
      sendResponse({ session: currentSession });
      break;

    case 'END_SESSION':
      // User finished searching
      currentSession = { query: null, startedAt: null, apiUrl: DEFAULT_API_URL, source: null, intent: null };
      browserAPI.contextMenus.update('saveSelectedAnswer', {
        title: 'Save as answer to: "%s"'
      });
      // Clear from storage
      saveSessionToStorage();
      // Broadcast to all tabs so they can hide the session bar
      broadcastToAllTabs({ type: 'SESSION_ENDED' });
      sendResponse({ success: true });
      break;

    case 'SAVE_PAGE_AS_ANSWER':
      // Save current page as answer
      saveAnswer({
        type: 'url',
        content: message.url,
        title: message.title
      }).then(result => sendResponse(result));
      return true; // Keep channel open for async response

    case 'SAVE_TEXT_AS_ANSWER':
      // Save selected text as answer
      saveAnswer({
        type: 'text_answer',
        content: message.text,
        title: message.context || `Selected from: ${message.url}`,
        url: message.url
      }).then(result => sendResponse(result));
      return true; // Keep channel open for async response

    case 'UPDATE_API_URL':
      currentSession.apiUrl = message.apiUrl;
      saveSessionToStorage();
      sendResponse({ success: true });
      break;
  }
});

// Save answer to the API
async function saveAnswer({ type, content, title, url }) {
  if (!currentSession.query) {
    return { success: false, error: 'No active query session' };
  }

  try {
    const response = await fetch(`${currentSession.apiUrl}/answers/submit`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        query_text: currentSession.query,
        answer_type: type,
        answer_content: content,
        answer_title: title,
        source_intent: 'fact'
      })
    });

    const data = await response.json();
    
    if (data.success) {
      return { success: true, message: 'Answer saved!' };
    } else {
      return { success: false, error: data.error || 'Failed to save' };
    }
  } catch (error) {
    console.error('[Background] Save error:', error);
    return { success: false, error: error.message };
  }
}

